using System;
using System.Threading;
using System.Runtime.InteropServices;

using gov.va.med.vbecs.Common;
using Microsoft.Win32.SafeHandles;
using gov.va.med.vbecs.ExceptionManagement;

namespace gov.va.med.vbecs.GUI.controls
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>3/18/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// Simple implementation of the regular Win32 mutex that differs from the standard FCL 
	/// System.Threading.Mutex implementation by providing access to Everyone by default. 
	/// This is needed to create global synchronization primitives shared between all VBECS users. 
	/// </summary>

	#endregion

	public class SharedWin32Mutex : WaitHandle
	{
		/// <summary>
		/// Win32 API security descriptor string defining MUTEX_ALL_ACCESS options.
		/// </summary>
		protected const string MUTEX_ALL_ACCESS_FOR_EVERYONE_STRING_SECURITY_DESCRIPTOR = "D:(A;NP;0x001f0001;;;WD)";

        private IntPtr _hMutex = IntPtr.Zero;

		/// <summary>
		/// Helper class encapsulating WinAPI calls/constants.
		/// </summary>
		protected class Win32Helper
		{
			/// <summary>
			/// SECURITY_ATTRIBUTES constant from WinAPI. See Platform SDK Reference.
			/// </summary>
			[StructLayout(LayoutKind.Sequential)]
				public struct SECURITY_ATTRIBUTES
			{
				/// <summary>
				/// nLength
				/// </summary>
				public int nLength;

				/// <summary>
				/// lpSecurityDescriptor
				/// </summary>
				public IntPtr lpSecurityDescriptor;

				/// <summary>
				/// bInheritHandle
				/// </summary>
				public bool bInheritHandle;
			}

			/// <summary>
			/// Creates mutex. See Platform SDK Reference.
			/// </summary>
			/// <param name="mutexAttributes">Mutex security attributes.</param>
			/// <param name="bInitialOwner">Specifies a method call should request initial ownership.</param>
			/// <param name="lpName">Name of the mutex.</param>
			/// <returns>Handle of the new/existing mutex.</returns>
			[DllImport("kernel32.dll", SetLastError=true)]
			public static extern IntPtr CreateMutex( ref SECURITY_ATTRIBUTES mutexAttributes, bool bInitialOwner, string lpName );



			/// <summary>
			/// Releases mutex. See Platform SDK Reference.
			/// </summary>
			/// <param name="hMutex">Handle of the mutex to release.</param>
			/// <returns>Success indicator.</returns>
			[DllImport("kernel32.dll")]
			public static extern bool ReleaseMutex( IntPtr hMutex );

			/// <summary>
			/// Frees the specified local memory object and invalidates its handle. See Platform SDK Reference.
			/// </summary>
			/// <param name="hMem">Handle to memory structure.</param>
			/// <returns>Error code.</returns>
			[DllImport("kernel32.dll")]
			public static extern uint LocalFree( IntPtr hMem );

			/// <summary>
			/// Converts a string-format security descriptor into a valid, functional security descriptor. See Platform SDK Reference.
			/// </summary>
			/// <param name="StringSecurityDescriptor">String security descriptor.</param>
			/// <param name="StringSDRevision">Specifies the revision level of the StringSecurityDescriptor string.</param>
			/// <param name="SecurityDescriptor">Pointer to a variable that receives a pointer to the converted security descriptor.</param>
			/// <param name="SecurityDescriptorSize">Pointer to a variable that receives the size, in bytes, of the converted security descriptor.</param>
			/// <returns>Error code.</returns>
			[DllImport("Advapi32.dll")]
			public static extern bool ConvertStringSecurityDescriptorToSecurityDescriptor( string StringSecurityDescriptor, UInt32 StringSDRevision, ref IntPtr SecurityDescriptor, ref int SecurityDescriptorSize );

			/// <summary>
			/// SYNCHRONIZE constant from WinAPI. See Platform SDK Reference.
			/// </summary>
			public const uint SYNCHRONIZE = 0x00100000;

			/// <summary>
			/// SDDL_REVISION_1 constant from WinAPI. See Platform SDK Reference.
			/// </summary>
			public const uint SDDL_REVISION_1 = 1;

			/// <summary>
			/// Error number constant from WinAPI. See Platform SDK Reference.
			/// </summary>
			public const uint ERROR_ALREADY_EXISTS = 183;
		}

		private bool _createdNew;

		/// <summary>
		/// Initializes mutex with a given name that is available for all users in Everyone group.
		/// </summary>
		/// <param name="name">Name for the mutex.</param>
		public SharedWin32Mutex( string name )
			: base()
		{
			if( name == null )
				throw( new ArgumentNullException( "name" ) );

            


			Win32Helper.SECURITY_ATTRIBUTES sa = CreateMutexForEveryOneSecurityAttributes();

            _hMutex = Win32Helper.CreateMutex(ref sa, false, name);

            

			_createdNew = ( Marshal.GetLastWin32Error() != Win32Helper.ERROR_ALREADY_EXISTS );

			Win32Helper.LocalFree( sa.lpSecurityDescriptor );

            if (_hMutex == IntPtr.Zero)
				throw new BaseApplicationException( StrRes.SysErrMsg.Common.UnableToCreateOrOpenWin32Mutex().ResString );
		}

		/// <summary>
		/// Initializes mutex with a given name that is available for all users in Everyone group.
		/// Checks if a new mutex was created or a reference to existing mutex was obtained. 
		/// </summary>
		/// <param name="name">Name for the mutex.</param>
		/// <param name="createdNew">
		///		Indicates if new mutex was created (true). 
		///		If a false is returned - it means that WinAPI call obtained a reference to existing mutex.
		///	</param>
		public SharedWin32Mutex( string name, out bool createdNew )
			: this( name )
		{
			createdNew = _createdNew;
		}

		/// <summary>
		/// Releases the mutex by calling appropriate WinAPI function.
		/// </summary>
		public void ReleaseMutex()
		{
            if (_hMutex == IntPtr.Zero)
				throw( new BaseApplicationException( StrRes.SysErrMsg.Common.UnableToReleaseWin32Mutex().ResString ) );

            Win32Helper.ReleaseMutex(_hMutex);
		}

		/// <summary>
		/// Creates and initializes an instance of security attributes 
		/// specifying access for everyone for new mutex.
		/// </summary>
		/// <returns>Instance of security attributes initialized with access for everyone for new mutex.</returns>
		protected Win32Helper.SECURITY_ATTRIBUTES CreateMutexForEveryOneSecurityAttributes()
		{
			Win32Helper.SECURITY_ATTRIBUTES sa = new Win32Helper.SECURITY_ATTRIBUTES();

			sa.nLength = Marshal.SizeOf( sa );
			sa.bInheritHandle = true;
			sa.lpSecurityDescriptor = IntPtr.Zero;

			int securityDescriptorSize = 0; // dummy

			bool result = Win32Helper.ConvertStringSecurityDescriptorToSecurityDescriptor(
				MUTEX_ALL_ACCESS_FOR_EVERYONE_STRING_SECURITY_DESCRIPTOR, 
				Win32Helper.SDDL_REVISION_1,
				ref sa.lpSecurityDescriptor,
				ref securityDescriptorSize );

			if( !result )
				throw new BaseApplicationException( StrRes.SysErrMsg.Common.FailedToCreateSecurityDescriptorForWin32Mutex().ResString );

			return sa;
		}
	}
}
